<?php
/**
 * Submit Form
 *
 * @package     AutomatorWP\Integrations\UserFeedback\Triggers\submit-survey
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_UserFeedback_Anonymous_Submit_Survey extends AutomatorWP_Integration_Trigger {

    public $integration = 'userfeedback';
    public $trigger = 'userfeedback_anonymous_submit_survey';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'anonymous'         => true,
            'label'             => __( 'User submits an anonymous survey', 'automatorwp-pro' ),
            'select_option'     => __( 'User submits <strong>an anonymous survey</strong>', 'automatorwp-pro' ),
            'edit_label'        => sprintf( __( 'User submits %1$s', 'automatorwp-pro' ), '{survey}' ),
            'log_label'         => sprintf( __( 'User submits %1$s', 'automatorwp-pro' ), '{survey}' ),
            'action'            => 'userfeedback_survey_response',
            'function'          => array( $this, 'listener' ),
            'priority'          => 9,
            'accepted_args'     => 3,
            'options'           => array(
                'survey' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'survey',
                    'name'              => __( 'Survey:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any survey', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_userfeedback_get_surveys',
                    'options_cb'        => 'automatorwp_userfeedback_options_cb_form',
                    'default'           => 'any'
                ) ),
            ),
            'tags' => array_merge(
                automatorwp_userfeedback_get_survey_tags()
            )
        ) ); 

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $survey_id
     * @param int $response_id
     * @param array $request
     */
    public function listener( $survey_id, $response_id, $request ) {

        $user_id = get_current_user_id();
        
        // Login is not required
        if ( $user_id !== 0 ) {
            return;
        }

        // Tags
        $survey_first_response = $request['answers'][0]['value'];
        $survey_all_repsonses = automatorwp_userfeedback_get_all_responses( $request['answers'] );
        $survey_title = automatorwp_userfeedback_get_survey_title($survey_id); 
        $survey_question = automatorwp_userfeedback_get_survey_question($survey_id);
        $user_ip = automatorwp_userfeedback_get_user_ip( $response_id );

        // Trigger submit form event
        automatorwp_trigger_event( array(
            'trigger'               => $this->trigger,
            'survey_id'             => $survey_id,
            'survey_title'          => $survey_title,
            'survey_first_response' => $survey_first_response,
            'survey_all_responses'  => $survey_all_repsonses,
            'survey_question'       => $survey_question,
            'user_ip'               => $user_ip,
        ) );

    }

    /**
     * Anonymous deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function anonymous_deserves_trigger( $deserves_trigger, $trigger, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['survey_id'] ) ) {
            return false;
        }

        // Bail if post doesn't match with the trigger option
        if( $trigger_options['survey'] !== 'any' && absint( $event['survey_id'] ) !== absint( $trigger_options['survey'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_anonymous_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['survey_title'] = ( isset( $event['survey_title'] ) ? $event['survey_title'] : '' );
        $log_meta['survey_question'] = ( isset( $event['survey_question'] ) ? $event['survey_question'] : '' );
        $log_meta['survey_first_response'] = ( isset( $event['survey_first_response'] ) ? $event['survey_first_response'] : '' );
        $log_meta['survey_all_responses'] = ( isset( $event['survey_all_responses'] ) ? $event['survey_all_responses'] : '' );
        $log_meta['user_ip'] = ( isset( $event['user_ip'] ) ? $event['user_ip'] : '' );
        $log_meta['survey_id'] = ( isset( $event['survey_id'] ) ? $event['survey_id'] : '' );
        
        return $log_meta;

    }

}

new AutomatorWP_UserFeedback_Anonymous_Submit_Survey();